<?php
declare(strict_types=1);
namespace App;
use PDO;

class DB {
    private static ?PDO $pdo = null;
    public static function pdo(): PDO {
        if (self::$pdo) return self::$pdo;
        $dsn = 'sqlite:' . __DIR__ . '/../storage/app.db';
        self::$pdo = new PDO($dsn, null, null, [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        ]);
        $sql = file_get_contents(__DIR__ . '/../migrations/schema.sql');
        self::$pdo->exec($sql);
        // ensure columns/tables (migrations souples)
        $cols = self::$pdo->query("PRAGMA table_info(documents)")->fetchAll();
        $names = array_map(fn($c)=>$c['name'],$cols);
        if (!in_array('token',$names,true)) self::$pdo->exec("ALTER TABLE documents ADD COLUMN token TEXT");
        if (!in_array('type',$names,true)) self::$pdo->exec("ALTER TABLE documents ADD COLUMN type TEXT");
        if (!in_array('expires_at',$names,true)) self::$pdo->exec("ALTER TABLE documents ADD COLUMN expires_at TEXT");
        // ensure vehicles table
        self::$pdo->exec("CREATE TABLE IF NOT EXISTS vehicles (vehicle_ref TEXT PRIMARY KEY, token TEXT, created_at TEXT)");
        // seed admin
        $count = (int) self::$pdo->query('SELECT COUNT(*) FROM users')->fetchColumn();
        if ($count === 0) {
            $email = $_ENV['ADMIN_EMAIL'] ?? 'admin@c-i-d-f.fr';
            $pass  = $_ENV['ADMIN_PASSWORD'] ?? 'change-moi';
            $hash  = password_hash($pass, PASSWORD_DEFAULT);
            $ins = self::$pdo->prepare('INSERT INTO users(email, password_hash, created_at) VALUES(?,?,?)');
            $ins->execute([$email, $hash, gmdate('c')]);
        }
        return self::$pdo;
    }

    public static function userByEmail(string $email): ?array {
        $st = self::pdo()->prepare('SELECT * FROM users WHERE email = ?'); $st->execute([$email]); return $st->fetch() ?: null;
    }

    public static function ensureVehicle(string $vehicle_ref): string {
        $vehicle_ref = trim($vehicle_ref);
        if ($vehicle_ref === '') return '';
        $st = self::pdo()->prepare('SELECT token FROM vehicles WHERE vehicle_ref = ?');
        $st->execute([$vehicle_ref]);
        $tok = $st->fetchColumn();
        if ($tok) return (string)$tok;
        $token = bin2hex(random_bytes(16));
        $ins = self::pdo()->prepare('INSERT INTO vehicles(vehicle_ref, token, created_at) VALUES(?,?,?)');
        $ins->execute([$vehicle_ref, $token, gmdate('c')]);
        return $token;
    }

    public static function vehicleToken(string $vehicle_ref): ?string {
        $st = self::pdo()->prepare('SELECT token FROM vehicles WHERE vehicle_ref = ?');
        $st->execute([$vehicle_ref]);
        $tok = $st->fetchColumn();
        return $tok ? (string)$tok : null;
    }

    public static function vehicle(string $vehicle_ref): ?array {
        $st = self::pdo()->prepare('SELECT * FROM vehicles WHERE vehicle_ref = ?');
        $st->execute([$vehicle_ref]);
        $row = $st->fetch();
        return $row ?: null;
    }

    public static function vehicleDocs(string $vehicle_ref): array {
        $st = self::pdo()->prepare('SELECT * FROM documents WHERE vehicle_ref = ? ORDER BY created_at DESC');
        $st->execute([$vehicle_ref]);
        return $st->fetchAll();
    }

    public static function insertDoc(array $d): void {
        $st = self::pdo()->prepare('INSERT INTO documents(id, vehicle_ref, filename, stored_name, mime, size, created_at, token, type, expires_at) VALUES(?,?,?,?,?,?,?,?,?,?)');
        $st->execute([$d['id'], $d['vehicle_ref'], $d['filename'], $d['stored_name'], $d['mime'], $d['size'], gmdate('c'), $d['token'] ?? null, $d['type'] ?? null, $d['expires_at'] ?? null]);
    }
    public static function updateToken(string $id, string $token): void {
        $st = self::pdo()->prepare('UPDATE documents SET token = ? WHERE id = ?'); $st->execute([$token, $id]);
    }
    public static function deleteDoc(string $id): void {
        $st = self::pdo()->prepare('DELETE FROM documents WHERE id = ?'); $st->execute([$id]);
    }
    public static function doc(string $id): ?array {
        $st = self::pdo()->prepare('SELECT * FROM documents WHERE id = ?'); $st->execute([$id]); return $st->fetch() ?: null;
    }
    public static function docs(): array {
        return self::pdo()->query('SELECT id, vehicle_ref, filename, stored_name, mime, size, created_at, token, type, expires_at FROM documents ORDER BY created_at DESC LIMIT 1000')->fetchAll();
    }
    public static function purgeAll(): void { self::pdo()->exec('DELETE FROM documents'); self::pdo()->exec('VACUUM'); }

    public static function docsExpiringWithin(array $daysList): array {
        $now = new \DateTimeImmutable('now', new \DateTimeZone('UTC'));
        $out = [];
        foreach ($daysList as $d) {
            $limit = $now->add(new \DateInterval('P'+(string)$d+'D')); // placeholder
        }
        // simpler: fetch all with expires_at not null, filter in PHP
        $rows = self::pdo()->query('SELECT * FROM documents WHERE expires_at IS NOT NULL')->fetchAll();
        return $rows;
    }
}
